/*

Dissertation - "Using Java To Implement Constraint Satisfaction Systems"
========================================================================

The Constraint Propagation Networks Client Program
--------------------------------------------------

Compatible With Java Version 1.1.3

*/

import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

public class NetworkClient extends Applet
{
  // keep record of any messages during last "addUserConstraint"
  public static StringBuffer netMessage = new StringBuffer();

  private InputHandler input = new InputHandler(this); // to enter expressions
  private TextArea outputArea = new TextArea(); // to display outputs
  private ConHandler conHdlr = new ConHandler(this);
  private VarHandler varHdlr = new VarHandler();

  private Parser par = new Parser();
  // table of operator constructors
  private TableOfConstructors csrTable =
                                     new TableOfConstructors();
  // for generating each part of network
  private NetGenerator netGen = new NetGenerator(varHdlr, csrTable);
 
  public void init()
  {
    resize(380,300);

    setLayout(new BorderLayout(10,0));

    // panel for list of constraints and variables
    Panel lists = new Panel();
    lists.setLayout(new GridLayout(1,2,20,0));
    lists.add("East", conHdlr);
    lists.add("West", varHdlr);

    // add and position components
    add("North", input);
    add("Center", lists);
    add("South", outputArea);

    // ** Make further additions to the operator and constructor tables
    // Centigrade - Farenheit Converter Constructor
    addConstructorDefinition("tempconv", "TempConvCsr", 2, 0);
    // Kelvin - Centigrade - Farenheit Converter Constructor
    addConstructorDefinition("tempconv2", "TempConvCsr2", 3, 0);
  }

  public void reset()
  {
    outputArea.append("\nNew constraint propagation network!\n");
    // remove constraints from display
    conHdlr.reset();
    // remove variables from display
    varHdlr.reset();
  }

  // ** try to add the constraint (represented by the string) to the network
  public void addUserConstraint(String str)
  {
    String conStr = new String(str);

    outputArea.append("Trying to add constraint...\n");

    Expression e = par.parse(conStr);

    if (e == null) {
      // ** string could not be parsed
      outputArea.append("String could not be parsed!\n");
    } else if (e.isConstraint()) {
      // ** string successfully parsed and is a constraint!
      
      // create and add new propagation network to rest of the existing network
      // ... and get the constructors which form this user constraint
      Vector newCsrs = netGen.generate((OprExpr) e);

      // add the new constraint to the list
      conHdlr.addCon(new UserConstraint(conStr, (OprExpr) e, newCsrs));
      
      // ** user constraint successfully parsed (and so added)
      conHdlr.addItem(conStr);
      outputArea.append("Constraint \"" +conStr+ "\" added.\n");
    } else {
      // ** string successfully parsed but is not a constraint
      outputArea.append("Expression is not a constraint!\n");
    }

    updateDisplays(); // show effect of changes to the network
  }

  public void removeUserConstraint(int index)
  {
    UserConstraint c = conHdlr.removeCon(index);

    outputArea.append("Removing constraint \"" + c.getName() + "\"\n");

    // get array of sub-constraints
    Vector csrs = c.getConstructors();

    // ** remove each constructor one at a time, in reverse order
    for (int i = csrs.size() - 1; i >= 0; i--) {
      Constructor csr = (Constructor) csrs.elementAt(i);
      // ** dont kill input constructors because they might be shared with
      // ** other constraints
      if (!(csr instanceof InputCsr)) {
        csr.kill();
      }
    }

    updateDisplays(); // show effect of changes to the network
  }

  // ** set or forget a variable's value
  public void changeValue(String varName, String valString)
  {
    // get corresponding variable constructor
    InputCsr csr = varHdlr.getVarConstructor(varName);

    if (csr == null) {
      // ** variable does not exist!
      outputArea.append("Variable not found!\n");
    } else if (valString.equals("?")) {
      // ** variable does exist and user want to lose its value
      csr.forgetValueFromOutside();
    } else {
      // ** variable does exist and user wants to set a "proper" value
      Integer objValue = Parser.getValue(valString);
      if (objValue != null) {
	// ** we have a valid value
	csr.setValueFromOutside(objValue.intValue());
      } else {
	outputArea.append("New value \"" + valString + "\" is not valid!\n");
      }
    }

    updateDisplays(); // show effect of changes to the network
  }


  public void unknownInput()
  {
    outputArea.append("\nUnknown input command!\n\n" +
		      "\"add <constraint expression>\" to add constraint\n" +
		      "\"<variable name> = <value>\" to set value" +
		      "\"<variable name> = ?\" to forget value" +
		      "\"new\" to clear the constraint network\n"+
		      "Double click on constraint to remove it.\n\n");
  }
  private void updateDisplays()
  {
    conHdlr.update();
    varHdlr.update();
    // show any messages from Network object
    if (netMessage.toString() != null) {
      outputArea.append(netMessage.toString());
      netMessage.setLength(0); // reset network message stringbuffer
    }
  }

  public void addConstructorDefinition
    (String name, String className, int arity, int pri)
  {
    // ** "name" is the string which appears in expression strings
    // ** "className" is the name of the class definiing the constructor
    // ** "arity" is the arity of the corresponding standard operator
    // ** "pri" is the priority of the corresponding standard operator

    TableOfOperators oprTable = par.getTableOfOperators(); //get operator table

    int oprType = oprTable.getNextFree(); // get new operator type number
    // update parser's table of operators
    oprTable.put(name, new Operator(oprType, arity, pri, true));
    // update table of constructors
    csrTable.put(oprType, className);
  }
}

class InputHandler extends TextField implements ActionListener
{
  private NetworkClient netClnt; // must know who to tell to remove constraint

  InputHandler(NetworkClient netClnt)
  {
    super();
    this.netClnt = netClnt;

    addActionListener(this);
  }

  public void actionPerformed(ActionEvent e)
  {
    // ** handle pressing return (ie entering string) in the text field
    String str = new String(e.getActionCommand().trim());

    if (str.equalsIgnoreCase("new")) {
      // ** clear old network so ready for new network
      netClnt.reset();
    } else if (str.startsWith("add ")) {
      // ** try to add user constraint to network
      netClnt.addUserConstraint(str.substring(4));
    } else if (str.indexOf("=") != -1){
      // ** assume user is trying to update a variable's value
      // tell Client the strings on either side of "=" sign - first part
      //   should be variable name, second part should be new value or "?"
      netClnt.changeValue((str.substring(0, str.indexOf("="))).trim(),
                          (str.substring(str.indexOf("=") + 1)).trim());
    } else {
      // ** unknown input
      netClnt.unknownInput();
    }      

    selectAll();
  }
}

class ConHandler extends List implements ActionListener
{
  private NetworkClient netClnt;
  private Vector userCons = new Vector(); // vector of all user constraints

  ConHandler(NetworkClient netClnt)
  {
    super(7, true);
    this.netClnt = netClnt;

    addActionListener(this);
  }

  // ** handle double clicking on a user constraint in the constraints list
  public void actionPerformed(ActionEvent e)
  {
    // ** get index of clicked user constraint
    int index = getSelectedIndex(); // use currently selected index by default
    if (index < 0) {
      // ** the item must have been deselected already
      String str = e.getActionCommand(); // get string item
      String [] items = getItems();
      for (int i = 0; i < items.length; i++) {
        if (str.equals(items[i])) {
          index = i;
          break;
        }
      }
    }

    if (index >= 0) {
      // remove selected user constraint from list
      netClnt.removeUserConstraint(index);
    }
  }

  public void reset()
  {
    removeAll(); // remove all items from list
    userCons.removeAllElements(); // no user constraints yet!
  }

  public void addCon(UserConstraint userCon)
  {
    userCons.insertElementAt(userCon, 0);
  }

  // remove user constraint with corres. "index" from the list of variables
  public UserConstraint removeCon(int index)
  {
    // ** remove selected constraint
    
    UserConstraint removed = (UserConstraint) userCons.elementAt(index);

    userCons.removeElementAt(index); // remove element from main vector

    update(); // update the visible list of user constraints

    return removed;
  }

  public void update()
  {
    String status; //string to hold whether user constraint is satisfied or not

    removeAll(); // clear old list
    for (int i = 0; i < userCons.size(); i++) {
      // get user constraint's output connector
      UserConstraint c = (UserConstraint) userCons.elementAt(i);

      if (c.isValid()) {
        // no contradiction
        status = "{t} ";
      } else {
        // a contradiction exists
        status = "{f} ";
      }

      addItem(status + c.getName());
    }
  }
}

class VarHandler extends List
{
  //match variable names to variable objects
  public Hashtable varHash = new Hashtable();

  VarHandler()
  { super(7, true); }

  public void reset()
  {
    removeAll(); // remove all variables from list
    varHash.clear(); // remove all variables from hashtable
  }

  public InputCsr addVar(String name)
  {
    InputCsr csr;

    // ** add this variable to the hashtable if it did not previously exist
    if (!varHash.containsKey(name)) {
      // ** variable doesnt already exist
      // make new input constructor
      csr = new InputCsr(new Connector(NetworkClient.netMessage),
			 name, NetworkClient.netMessage);
      // add new input constructor to hash table
      varHash.put(name, csr);
    } else {
      // ** variable already exists
      // get input constructor
      csr = (InputCsr) varHash.get(name);
    }

    return csr;
  }

  public InputCsr getVarConstructor(String varName)
  { return (InputCsr) varHash.get(varName); }

  public void update()
  {
    removeAll(); // clear old list

    for (Enumeration e = varHash.keys() ; e.hasMoreElements() ;) {
      String name = (String) e.nextElement();
      InputCsr csr = (InputCsr) varHash.get(name);
      if (csr.isDead()) {
        // ** the constructor's connector has no other constructor joined to it
        // ** and so this constructor does not participate in the network
        // ** so remove it!
        varHash.remove(name);
      } else {
        // ** the constructor still has an active role in the network
        // ** so show it!
        // does variable have a value
        if (csr.hasValue()) {
          // we have variable name and value
          addItem(name + " = " + csr.getValue());       
        } else {
          // we have variable name and variable has no value
          addItem(name + " = ?");
        }
      }
    }
  }
}

class UserConstraint
{
  private String name;
  private OprExpr e;
  private Vector csrs;

  UserConstraint(String name, OprExpr e, Vector csrs)
  {
    this.name = name;
    this.e = e;
    this.csrs = csrs;
  }

  public String getName()
  { return name; }

  public OprExpr getExpression()
  { return e; }
  
  public Vector getConstructors()
  { return csrs; }

  public boolean isValid()
  {
    // ** 
    for (int i = 0; i < csrs.size(); i++) {
      Constructor csr = (Constructor) csrs.elementAt(i);
      if (!csr.isValid()) {
        return false;
      }
    }
    return true;
  }
}

// ** Hashtable subclass which keeps a record of the constructor class which
// ** correspond to each type of Operator
class TableOfConstructors extends Hashtable
{
  TableOfConstructors()
  {
    super();

    // ** relational operators
    put(TableOfOperators.EQU, "EqualityCsr");
    put(TableOfOperators.UNE, "UnequalityCsr");
    put(TableOfOperators.LES, "LessThanCsr");
    put(TableOfOperators.GRE, "GreaterThanCsr");
    put(TableOfOperators.LESEQU, "LessThanOrEqualCsr");
    put(TableOfOperators.GREEQU, "GreaterThanOrEqualCsr");
    // ***** Normal Unary Operators  *****
    put(TableOfOperators.FAC, "FactorialCsr");
    put(TableOfOperators.NEG, "NegationCsr");
    // ***** Normal Binary Operators *****
    put(TableOfOperators.ADD, "AdditionCsr");
    put(TableOfOperators.SUB, "SubtractionCsr");
    put(TableOfOperators.MUL, "MultiplicationCsr");
    put(TableOfOperators.DIV, "DivisionCsr");
    put(TableOfOperators.MAX, "MaximumCsr");
    put(TableOfOperators.MIN, "MinimumCsr");
  }

  public boolean put(int oprType, String className)
  {
    try {
      super.put(new Integer(oprType), Class.forName(className));
    }
    catch (ClassNotFoundException e) {
      System.out.println
        ("Unable to find constructor class \"" + className + "\"!");
      return false; // addition failed
    }
    return true; // addition successful
  }
}

class NetGenerator implements Visitor
{
  // object containing hashtable of all constructors
  private TableOfConstructors csrTable;

  private VarHandler varHdlr;
  private Vector newCsrs;

  NetGenerator(VarHandler varHdlr, TableOfConstructors csrTable)
  {
    this.varHdlr = varHdlr;
    this.csrTable = csrTable;
  }

  public Vector generate(OprExpr e)
  {
    newCsrs = new Vector(); // create new Vector to hold new constructors made

    // Traverse the tree - the first connector in cnrs will be the ouput of
    // the main relational operator of the constraint
    e.visit(this);    

    return newCsrs;
  }

  public Object visitVar(VarExpr e)
  {
    // ** visiting a variable expression

    // ** get constructor for this variable
    InputCsr csr = varHdlr.addVar(e.getName());
    // update list of constructors for this user constraint
    newCsrs.addElement(csr);

    // return the variable connector
    return csr.getConnector();
  }

  public Object visitNum(NumExpr e)
  {
    // ** visiting a constant number expression

    // make new output connector (via network handler)
    Connector cnr = new Connector(NetworkClient.netMessage);
    // make a new constant number constructor 
    newCsrs.addElement
      (new ConstantCsr(cnr, e.getValue(), NetworkClient.netMessage));
    // return the "loose" output connector
    return cnr;
  }

  // ** visiting an operation expression (also handles relational ones)
  public Object visitOpr(OprExpr e)
  {
    int opType = e.getOp().getType(); // type of operator
    int opArity = e.getOp().getArity(); // arity of operator

    Connector [] cnrs; // array of connectors
    Connector cnr; // output connector, for if there is one
    // ** make array of connectors
    if (e.getOp().isRelational()) {
      // ** operator is relational so we dont have an output connector
      cnrs = new Connector[opArity];
      cnr = null; // no loose outout connector
    } else {
      // ** operator is not relational so we have an output connector
      cnrs = new Connector[opArity + 1];
      // ** loose "output" connector
      cnr = new Connector(NetworkClient.netMessage);
      cnrs[opArity] = cnr;
    }
    // ** get the connectors to all its inputs
    for (int i = 0; i < opArity; i++) {
      cnrs[i] = (Connector) ((e.getExprs())[i]).visit(this);
    }

    // ** make a new constructor corresponding to this operator **

    Integer opTypeInteger = new Integer(opType);

    if (csrTable.containsKey(opTypeInteger)) {
      // ** this operator does indeed have a constructor class for it
      try {
        OperatorConstructor csr = (OperatorConstructor)
          ((Class) csrTable.get(opTypeInteger)).newInstance();
        csr.init(cnrs, NetworkClient.netMessage);
        newCsrs.addElement(csr);
      }
      catch (Exception except) { System.out.println("Error Finding Class"); }
    }

    // return the "loose" output connector, if there is one
    return cnr;
  }
}

// ** Example Constructors **

// ** "9 * cnrs[0] = 5 * (cnrs[1] - 32)"
class TempConvCsr extends OperatorConstructor
{
  public void processNewValue()
  {
    if (cnrs[0].hasValue()) {
      cnrs[1].setValue((9 * cnrs[0].getValue()) / 5 + 32, this);
    } else if (cnrs[1].hasValue()) {
      cnrs[0].setValue((5 * (cnrs[1].getValue() - 32)) / 9, this);
    }
  }
}

// ** "tempconv k c f"
// ** "cnrs[0] = cnrs[1] + 273, 9 * cnrs[1] = 5 * (cnrs[2] - 32)"
class TempConvCsr2 extends OperatorConstructor
{
  public void processNewValue()
  {
    if (cnrs[0].hasValue()) {
      cnrs[1].setValue(cnrs[0].getValue() - 273, this);
      cnrs[2].setValue((9 * (cnrs[0].getValue() - 273)) / 5 + 32, this);
    } else if (cnrs[1].hasValue()) {
      cnrs[0].setValue(cnrs[1].getValue() + 273, this);
      cnrs[2].setValue((9 * cnrs[1].getValue()) / 5 + 32, this);
    } else if (cnrs[2].hasValue()) {
      cnrs[0].setValue((5 * (cnrs[2].getValue() - 32)) / 9 + 273, this);
      cnrs[1].setValue((5 * (cnrs[2].getValue() - 32)) / 9, this);
    }
  }
}


/*

The End

*/
