/*

Dissertation - "Using Java To Implement Constraint Satisfaction Systems"
========================================================================

Implementation Of Constraint Propagation Networks
-------------------------------------------------

Uses Java Version 1.1.3

*/

import java.util.*;

// *************************************************************************
// *******     P R O P A G A T I O N    N E T W O R K    C O D E    ********
// *************************************************************************

class Connector
{
  private int value;
  private Constructor informant;
  private Vector csrs = new Vector();
  private StringBuffer netMessage; // stringbuffer we can append messages to

  Connector(StringBuffer netMessage)
  { this.netMessage = netMessage; }

  public boolean hasValue()
  { return (informant != null); }

  public int getValue()
  { return value; }

  public boolean setValue(int newValue, Constructor setter)
  {
    if (!hasValue()) {
      // ** connector doesnt already have a value
      value = newValue; // get the new value
      informant = setter; // remember who told connector to set value
      // go through all connected constructors except for the informant
      for (int i = 0; i < csrs.size(); i++) {
        // get next constructor in the list
        Constructor c = (Constructor) csrs.elementAt(i);
        if (!c.equals(informant)) {
          c.processNewValue(); // tell constructor to process new value
        }
      }
      return true;
    } else if (newValue != value) {
      // ** connector already has a value which is not the same as new value
      netMessage.append("Error! Contradiction (" +value +" " +newValue +")\n");
      return false;
    }
    return true;
  }
  
  public void forgetValue(Constructor retractor)
  {
    if (hasValue() && retractor.equals(informant)) {
      // ** the constructor which informed the connector of the current
      // ** value, is the one which is trying to retract it

      informant = null; // connector no longer has an informant
      // go through all connected constructors except for the informant
      for (int i = 0; i < csrs.size(); i++) {
        // get next constructor in the list
        Constructor c = (Constructor) csrs.elementAt(i);
        if (!c.equals(retractor))
          c.processForgetValue(); // tell constructor to process new value
      }
    }
  }

  public void connect(Constructor newCsr)
  {
    if (!csrs.contains(newCsr))
      csrs.addElement(newCsr);
    if (hasValue())
      newCsr.processNewValue();
  }

  public void disconnect(Constructor oldCsr)
  {
    // ** pretend constructor has told me to forget my value
    //(we only need to remove first element because all elements are distinct)
    csrs.removeElement(oldCsr);
  }

  public boolean isLoose()
  { return csrs.size() < 2; }
}

// **********************  C O N S T R U C T O R S  *************************

abstract class Constructor
{
  protected Connector [] cnrs;
  protected boolean isValid = true;
  protected StringBuffer netMessage; // stringbuffer we can append messages to

  abstract public void processNewValue();
  abstract public void processForgetValue();

  public void kill()
  {
    // ** first disconnect all connectors from constructor
    // ** (must disconnect all before processing values because otherwise
    // **  values might get propagated back to this constructor)
    informDisconnect();

    // ** next tell them all to forget their value
    informForgetValue();
  }

  public boolean isValid()
  { return isValid; }

  protected void informConnect()
  {
    for (int i = 0; i < cnrs.length; i++) {
      cnrs[i].connect(this);
    }
  }

  protected void informDisconnect()
  {
    for (int i = 0; i < cnrs.length; i++) {
      cnrs[i].disconnect(this);
    }
  }

  protected void informForgetValue()
  {
    for (int i = 0; i < cnrs.length; i++) {
      cnrs[i].forgetValue(this);
    }    
  }
}

class InputCsr extends Constructor
{
  private String name;

  InputCsr(Connector cnr, String name, StringBuffer netMessage)
  {
    cnrs = new Connector[1];
    cnrs[0] = cnr;
    this.name = name;
    this.netMessage = netMessage;

    // tell the connector that it is connected to this constant constructor
    informConnect();
  }

  public void processNewValue()
  {
    netMessage.append("Network Setting: "+name+" = "+cnrs[0].getValue()+"\n");
  }

  public void processForgetValue()
  {
    netMessage.append("Network Forgetting: " + name + " = ?\n");    
  }

  public void setValueFromOutside(int value)
  {
    netMessage.append("User Setting: " + name + " = " + value + "\n");
    isValid = cnrs[0].setValue(value, this);
  }

  public void forgetValueFromOutside()
  {
    netMessage.append("User Forgetting: " + name + " = ?\n");    
    cnrs[0].forgetValue(this);
    isValid = true;
  }

  public boolean hasValue()
  { return cnrs[0].hasValue(); }

  public int getValue()
  { return cnrs[0].getValue(); }

  public Connector getConnector()
  { return cnrs[0]; }

  public boolean isDead()
  { return cnrs[0].isLoose(); }
}

class ConstantCsr extends Constructor
{
  private int value;

  ConstantCsr(Connector cnr, int value, StringBuffer netMessage)
  {
    // get the value of the constant and get the connector
    cnrs = new Connector[1];
    cnrs[0] = cnr;
    this.value = value;
    this.netMessage = netMessage;

    // tell the connector that it is connected to this constant constructor
    informConnect();
    isValid = cnrs[0].setValue(value, this);
  }

  public void processNewValue()
  {
    netMessage.append
      ("Error! Trying to process value for constant: " + value + ".\n");
  }

  public void processForgetValue()
  {
    isValid = cnrs[0].setValue(value, this);
  }
}

// ** Hashtable subclass which keeps a record of the constructor class which
// ** correspond to each type of Operator
class TableOfOperatorConstructors extends Hashtable
{
  TableOfOperatorConstructors()
  {
    // ** relational operators
    put(TableOfOperators.EQU, "EqualityCsr");
    put(TableOfOperators.UNE, "UnequalityCsr");
    put(TableOfOperators.LES, "LessThanCsr");
    put(TableOfOperators.GRE, "GreaterThanCsr");
    put(TableOfOperators.LESEQU, "LessThanOrEqualCsr");
    put(TableOfOperators.GREEQU, "GreaterThanOrEqualCsr");
    // ***** Normal Unary Operators  *****
    put(TableOfOperators.FAC, "FactorialCsr");
    put(TableOfOperators.NEG, "NegationCsr");
    // ***** Normal Binary Operators *****
    put(TableOfOperators.ADD, "AdditionCsr");
    put(TableOfOperators.SUB, "SubtractionCsr");
    put(TableOfOperators.MUL, "MultiplicationCsr");
    put(TableOfOperators.DIV, "DivisionCsr");
    put(TableOfOperators.MAX, "MaximumCsr");
    put(TableOfOperators.MIN, "MinimumCsr");
  }

  public boolean put(int oprType, String className)
  {
    try {
      super.put(new Integer(oprType), Class.forName(className));
    }
    catch (ClassNotFoundException e) {
      System.out.println
        ("Unable to find constructor class \"" + className + "\"!");
      return false; // addition failed
    }
    return true; // addition successful    
  }
}

abstract class OperatorConstructor extends Constructor
{
  public void init(Connector [] cnrs, StringBuffer netMessage)
  {
    this.cnrs = cnrs;
    this.netMessage = netMessage;
    informConnect();
  }

  public void processForgetValue()
  {
    isValid = true;
    informForgetValue(); // inform all connectors to "forget value"
    processNewValue();
  }
}

// ** cnrs[0] = cnrs[1]
class EqualityCsr extends OperatorConstructor
{
  public void processNewValue()
  {
    if (cnrs[0].hasValue()) {
      isValid = cnrs[1].setValue(cnrs[0].getValue(), this);
    } else if (cnrs[1].hasValue()) {
      isValid = cnrs[0].setValue(cnrs[1].getValue(), this);
    } else {
      isValid = true;
    }
  }
}

// ** cnrs[0] != cnrs[1]
class UnequalityCsr extends OperatorConstructor
{
  public void processNewValue()
  {
    if (cnrs[0].hasValue() && cnrs[1].hasValue()) {
      // both inputs have values
      if (cnrs[0].getValue() != cnrs[1].getValue()) {
        // inputs have different value - so inequality is satisfied
        isValid = true;
      } else {
        // inputs have same values - so inequality not satisfied
        isValid = false;
      }
    } else {
      // at least one input does not have a value
      isValid = true;
    }
  }
}

  // ** cnrs[0] < cnrs[1]
class LessThanCsr extends OperatorConstructor
{
  public void processNewValue()
  {
    if (cnrs[0].hasValue() && cnrs[1].hasValue()) {
      // both inputs have values
      if (cnrs[0].getValue() < cnrs[1].getValue()) {
        isValid = true;
      } else {
        isValid = false;
      }
    } else {
      // at least one input does not have a value
      isValid = true;
    }
  }
}

// ** cnrs[0] > cnrs[1]
class GreaterThanCsr extends LessThanCsr
{
  public void init(Connector [] cnrs, StringBuffer netMessage)
  {
    // ** "cnrs[0] > cnrs[1]" is like "cnrs[1] < cnrs[0]"
    // swap connectors 0 and 1...
    Connector dummy = cnrs[0];
    cnrs[0] = cnrs[1];
    cnrs[1] = dummy;
    // ... and use the LessThanCsr constructor
    super.init(cnrs, netMessage);
  }
}

// ** cnrs[0] <= cnrs[1] (just like greater than but valid is negated
class LessThanOrEqualCsr extends GreaterThanCsr
{
  public boolean isValid()
  { return !isValid; }
}

// ** cnrs[0] >= cnrs[1]
class GreaterThanOrEqualCsr extends LessThanOrEqualCsr
{
  public void init(Connector [] cnrs, StringBuffer netMessage)
  {
    // ** "cnrs[0] >= cnrs[1]" is like "cnrs[1] <= cnrs[0]"
    // swap connectors 0 and 1...
    Connector dummy = cnrs[0];
    cnrs[0] = cnrs[1];
    cnrs[1] = dummy;
    // ... and use the LessThanCsr constructor
    super.init(cnrs, netMessage);
  }
}

// ** cnrs[0]! = cnrs[1] 
class FactorialCsr extends OperatorConstructor
{
  public void processNewValue()
  {
    if (cnrs[0].hasValue()) {
      int value = 1;
      for (int i = 1; i <= cnrs[0].getValue(); i++) {
        value *= i;
      }
      isValid = cnrs[1].setValue(value, this);
    } else if (cnrs[1].hasValue()) {
      isValid = false;
      int value = cnrs[1].getValue();
      if (value >= 1) {
        int i;
        for (i = 1; value % i == 0; i++) {
          value /= i;
        }
        if (value == 1) {
          // the inverse factorial does indeed exist!
          isValid = cnrs[0].setValue(i - 1, this);
        }
      }
    } else {
      isValid = true;
    }
  }
}

// ** - cnrs[0] = cnrs[1]
class NegationCsr extends OperatorConstructor
{
  public void processNewValue()
  {
    if (cnrs[0].hasValue()) {
      isValid = cnrs[1].setValue(-cnrs[0].getValue(), this);
    } else if (cnrs[1].hasValue()) {
      isValid = cnrs[0].setValue(-cnrs[1].getValue(), this);
    } else {
      isValid = true;
    }
  }
}

// ** cnrs[0] + cnrs[1] = cnrs[2]
class AdditionCsr extends OperatorConstructor
{
  public void processNewValue()
  {
    if (cnrs[0].hasValue() && cnrs[1].hasValue())
      isValid =cnrs[2].setValue(cnrs[0].getValue() + cnrs[1].getValue(), this);
    else if (cnrs[0].hasValue() && cnrs[2].hasValue())
      isValid =cnrs[1].setValue(cnrs[2].getValue() - cnrs[0].getValue(), this);
    else if (cnrs[1].hasValue() && cnrs[2].hasValue())
      isValid =cnrs[0].setValue(cnrs[2].getValue() - cnrs[1].getValue(), this);
    else
      isValid = true;
  }
}

// ** cnrs[0] - cnrs[1] = cnrs[2]
class SubtractionCsr extends AdditionCsr
{
  public void init(Connector [] cnrs, StringBuffer netMessage)
  {
    // ** "cnrs[0] - cnrs[1] = cnrs[2]" is like "cnrs[2] + cnrs[1] = cnrs[0]"
    // swap connectors 0 and 2...
    Connector dummy = cnrs[0];
    cnrs[0] = cnrs[2];
    cnrs[2] = dummy;
    // ... and use the addition constructor
    super.init(cnrs, netMessage);
  }
}

// ** cnrs[0] * cnrs[1] = cnrs[2]
class MultiplicationCsr extends OperatorConstructor
{
  public void processNewValue()
  {
    if ((cnrs[0].hasValue() && cnrs[0].getValue() == 0) ||
        (cnrs[1].hasValue() && cnrs[1].getValue() == 0))
      isValid =cnrs[2].setValue(0, this);
    else if (cnrs[0].hasValue() && cnrs[1].hasValue())
      isValid =cnrs[2].setValue(cnrs[0].getValue() * cnrs[1].getValue(), this);
    else if (cnrs[2].hasValue() && cnrs[0].hasValue() && cnrs[0].getValue()!=0)
      isValid =cnrs[1].setValue(cnrs[2].getValue() / cnrs[0].getValue(), this);
    else if (cnrs[2].hasValue() && cnrs[1].hasValue() && cnrs[1].getValue()!=0)
      isValid =cnrs[0].setValue(cnrs[2].getValue() / cnrs[1].getValue(), this);
    else
      isValid = true;
  }
}

// ** cnrs[0] / cnrs[1] = cnrs[2]
class DivisionCsr extends MultiplicationCsr
{
  public void init(Connector [] cnrs, StringBuffer netMessage)
  {
    // ** "cnrs[0] / cnrs[1] = cnrs[2]" is like "cnrs[2] * cnrs[1] = cnrs[0]"
    // swap connectors 0 and 2...
    Connector dummy = cnrs[0];
    cnrs[0] = cnrs[2];
    cnrs[2] = dummy;
    // ... and use the multiplication constructor
    super.init(cnrs, netMessage);
  }
}

// ** max (cnrs[0], cnrs[1]) = cnrs[2]
class MaximumCsr extends OperatorConstructor
{
  public void processNewValue()
  {
    if (cnrs[0].hasValue() && cnrs[1].hasValue()) {
      isValid =
        cnrs[2].setValue(Math.max(cnrs[0].getValue(),cnrs[1].getValue()),this);
    } else if (cnrs[0].hasValue() && cnrs[2].hasValue()) {
      if (cnrs[2].getValue() >= cnrs[0].getValue()) {
        isValid = cnrs[1].setValue(cnrs[2].getValue(), this);
      } else {
        // error - max cant be smaller than one of its inputs
        isValid = false;
        netMessage.append
          ("Error! Contradiction - Max cannot be less than either input.\n");
      }
    } else if (cnrs[1].hasValue() && cnrs[2].hasValue()) {
      if (cnrs[2].getValue() >= cnrs[1].getValue()) {
        isValid = cnrs[0].setValue(cnrs[2].getValue(), this);
      } else {
        // error - max cant be smaller than one of its inputs
        isValid = false;
        netMessage.append
          ("Error! Contradiction - Max cannot be less than either input.\n");
      }
    }
  }
}

// ** min (cnrs[0], cnrs[1]) = cnrs[2]
class MinimumCsr extends OperatorConstructor
{
  public void processNewValue()
  {
    if (cnrs[0].hasValue() && cnrs[1].hasValue()) {
      isValid =
        cnrs[2].setValue(Math.max(cnrs[0].getValue(),cnrs[1].getValue()),this);
    } else if (cnrs[0].hasValue() && cnrs[2].hasValue()) {
      if (cnrs[2].getValue() <= cnrs[0].getValue()) {
        isValid = cnrs[1].setValue(cnrs[2].getValue(), this);
      } else {
        // error - min cant be bigger than one of its inputs
        isValid = false;
        netMessage.append
          ("Error! Contradiction - Min cannot be bigger than either input.\n");
      }
    } else if (cnrs[1].hasValue() && cnrs[2].hasValue()) {
      if (cnrs[2].getValue() <= cnrs[1].getValue()) {
        isValid = cnrs[0].setValue(cnrs[2].getValue(), this);
      } else {
        // error - min cant be bigger than one of its inputs
        isValid = false;
        netMessage.append
          ("Error! Contradiction - Min cannot be bigger than either input.\n");
      }
    }
  }
}


/*

The End

*/
