/*

Dissertation - "Using Java To Implement Constraint Satisfaction Systems"
========================================================================

The DeltaBlue Algorithm Client Program
--------------------------------------

Compatible With Java Version 1.1.3

*/

import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

public class DeltaBlueClient extends Applet implements ActionListener
{
  private InputHandler inputHdlr = new InputHandler(this); //enter expressions
  private StrengthHandler strengthHdlr = new StrengthHandler(); // set strength
  private TextArea outputArea = new TextArea(); // to display outputs
  private ConHandler conHdlr = new ConHandler(this);
  private VarHandler varHdlr = new VarHandler();

  private Parser par = new Parser();
  private TableOfConstraints conTable = new TableOfConstraints();
  private ConGenerator conGen = new ConGenerator(varHdlr, conTable);
  private DeltaBlue db = new DeltaBlue();

  private Plan currentPlan = null;

  public void init()
  {
    resize(380,300);

    setLayout(new BorderLayout(10,0));

    // panel for text fields at top of applet
    Panel inputs = new Panel();
    Button executeBut = new Button("Execute Plan");
    inputs.setLayout(new BorderLayout());
    inputs.add("West", executeBut);
    inputs.add("Center", inputHdlr);
    inputs.add("East", strengthHdlr);

    // Panel for constraints (constraints list and extract plan button)
    Panel conPanel = new Panel();
    Button extractConBut = new Button("Extract Plan From Constraints");
    conPanel.setLayout(new BorderLayout());
    conPanel.add("Center", conHdlr);
    conPanel.add("South", extractConBut);
    // Panel for variables (variables list and extract plan button)
    Panel varPanel = new Panel();
    Button extractVarBut = new Button("Extract Plan From Variables");
    varPanel.setLayout(new BorderLayout());
    varPanel.add("Center", varHdlr);
    varPanel.add("South", extractVarBut);

    // panel for panel of constraints and panel of variables
    Panel lists = new Panel();
    lists.setLayout(new GridLayout(1,2,20,0));
    lists.add("East", conPanel);
    lists.add("West", varPanel);

    // add and position components
    add("North", inputs);
    add("Center", lists);
    add("South", outputArea);

    outputArea.setEditable(false); // we dont want the user editing the output

    // ** add appropriate action listeners
    executeBut.addActionListener(this);
    extractConBut.addActionListener(this);
    extractVarBut.addActionListener(this);

    // ** Make further additions to the operator and constraint tables
    // Stay Constraint
    addConstraintDefinition("stay", "StayCon", 1, 0);
    // Edit Constraint
    addConstraintDefinition("edit", "EditCon", 1, 0);
    // Centigrade - Farenheit Converter Constraint
    addConstraintDefinition("tempconv", "TempConvCon", 2, 0);
  }

  public void actionPerformed(ActionEvent e)
  {
    // ** handle clicking a button
    String str = e.getActionCommand();

    if (str.equals("Execute Plan")) {
      // ** user clicked on "Execute Plan"
      executePlan();
    } else if (str.equals("Extract Plan From Constraints")) {
      // ** user clicked on "Extract Plan From Constraints"
      extractPlanFromConstraints();
    } else if (str.equals("Extract Plan From Variables")) {
      // ** user clicked on "Extract Plan From Variables"
      extractPlanFromVariables();      
    }
  }

  public void executePlan()
  {
    if (currentPlan != null) {
      // ** we do indeed have a plan to execute
      outputArea.append("Executing Current Plan.\n");

      db.executePlan(currentPlan);
      // update the list of variables and their values
      varHdlr.update();
    } else {
      // ** no plan to execute
      outputArea.append
	("No plan to execute! - You must extract a plan first!\n");
    }
  }

  public void extractPlanFromConstraints()
  {
    outputArea.append("Extracting Plan From Constraints.\n");

    currentPlan = db.extractPlan(conHdlr.getSelectedCons());
  }

  public void extractPlanFromVariables()
  {
    outputArea.append("Extracting Plan From Variables.\n");

    currentPlan = db.extractPlan(varHdlr.getSelectedVars());
  }

  public void reset()
  {
    outputArea.append("\nNew constraint hierachy!\n");
    varHdlr.reset();
    conHdlr.reset();
  }

  // ** try to add a user constraint (may be made up of many sub-constraints)
  public void addUserConstraint(String str)
  {
    outputArea.append("Trying to add constraint \"" + str + "\"...");

    Expression e = par.parse(str);

    if (e == null) {
      // ** string could not be parsed
      outputArea.append("String could not be parsed!\n");
    } else if (e.isConstraint()) {
      // ** string successfully parsed and is a constraint!

      // get strength of input constraint
      int strength = strengthHdlr.getStrength();

      // ** get an array of the sub-constraints corresponding to the
      // ** expression e, and given strength
      Vector cs = conGen.getSubConstraints(e, strength);
          
      //System.out.println("Formed " + cs.size() + " sub-constraints");

      // any message returned is an error message
      String errMessage = "";

      // ** add each sub-constraint one at a time (break on an exception)
      for (int i = 0; i < cs.size(); i++) {
        // tell the algorithm to add this constraint (get any errors)
        errMessage = db.addConstraint((Constraint) cs.elementAt(i));
        if (!errMessage.equals("")) {
          // ** could not add this sub-constraint
          // ** remove sub-constraints that have already been added
          for (int j = i - 1; j >= 0; j--) {
            db.removeConstraint((Constraint) cs.elementAt(j));
          }
          // show error message
          outputArea.append("Error - " + errMessage + "\n");
          break;
        }
      }

      if (errMessage.equals("")) {
        // ** no problem adding all sub-constraints
        // add the new input constraint to the list
        conHdlr.addCon(new UserConstraint(str, cs, strength));
        // plan no longer valid
        currentPlan = null;
        outputArea.append("Done.\n");
      }

      // update the list of variables and their values
      varHdlr.update();
    } else {
      // ** is not a relational expression (ie not a constraint)
      outputArea.append("Expression is not a constraint!\n");
    }
  }

  // ** remove a user constraint
  public void removeUserConstraint(int index)
  {
    UserConstraint c = conHdlr.removeCon(index);

    outputArea.append("Removing constraint \"" + c.getName() + "\"\n");

    // get array of sub-constraints
    Vector cs = c.getConstraints();

    // ** remove each sub-constraint one at a time, in reverse order
    for (int i = cs.size() - 1; i >= 0; i--) {
      db.removeConstraint((Constraint) cs.elementAt(i));
    }

    // plan no longer valid
    currentPlan = null;

    // update the list of variables and their values
    varHdlr.update();
  }

  // ** change the value used by the first selected edit constraint
  public void changeValue(String valString)
  {
    Integer objValue = Parser.getValue(valString);

    if (objValue != null) {
      // ** we have a legal value on the rhs!
      int value = objValue.intValue();
      if (conHdlr.changeValue(value)) {
	// ** selected input constraint updated
	outputArea.append
	  ("Edit constraint updated with value: \"" + value + "\"\n");
      } else {
	// ** variable does not exist and so value not set
	outputArea.append
	  ("No edit constraint selected!\n");
      }
    } else {
      // ** we have an illegal value on the rhs!
      outputArea.append("Illegal value on right hand side!\n");
    }

    varHdlr.update(); // show effect of changes to the network    
  }

  public void unknownInput()
  {
    outputArea.append("\nUnknown input command!\n\n" +
		      "\"add <constraint expression>\" to add constraint\n" +
		      "\"add edit <variable name>\" to add edit constraint\n"+
		      "\"change <value>\" to change value used by " +
		                        "selected edit constraint\n" +
		      "\"new\" to clear the constraint network\n"+
		      "Double click on constraint to remove it.\n\n");
  }

  public void addConstraintDefinition
    (String name, String className, int arity, int pri)
  {
    // ** "name" is the string which appears in expression strings
    // ** "className" is the name of the class defining the constraint
    // ** "arity" is the arity of the corresponding standard operator
    // ** "pri" is the priority of the corresponding standard operator

    TableOfOperators oprTable = par.getTableOfOperators(); //get operator table

    int oprType = oprTable.getNextFree(); // get new operator type number
    // update parser's table of operators
    oprTable.put(name, new Operator(oprType, arity, pri, true));
    // update table of constraints
    conTable.put(oprType, className);
  }
}

class InputHandler extends TextField implements ActionListener
{
  private DeltaBlueClient deltaClnt;

  InputHandler(DeltaBlueClient deltaClnt)
  {
    super();
    this.deltaClnt = deltaClnt;

    addActionListener(this);
  }

  public void actionPerformed(ActionEvent e)
  {
    // ** handle pressing return (ie entering string) in the text field
    String str = e.getActionCommand();
    
    if (str.equalsIgnoreCase("new")) {
      // ** clear old network so ready for new network
      deltaClnt.reset();
    } else if (str.startsWith("add ")) {
      // ** try to add constraint to network
      deltaClnt.addUserConstraint(str.substring(4).trim());
    } else if (str.startsWith("change ")) {
      // ** change the value used by an edit constraint
      deltaClnt.changeValue(str.substring(7).trim());
    } else {
      // ** unknown input
      deltaClnt.unknownInput();
    }

    selectAll();
  }
}

class StrengthHandler extends Panel implements ItemListener
{
  // ** strengths (WEAKEST and REQUIRED) are as in DeltaBlue algorithm
  // ** the others are spread out between them
  public static int WEAKEST = DeltaBlue.WEAKEST;
  public static int REQUIRED = DeltaBlue.REQUIRED;
  public static int MEDIUM = (REQUIRED + WEAKEST)/2;
  public static int WEAK = (MEDIUM + WEAKEST)/2;
  public static int STRONG = (REQUIRED + MEDIUM)/2;

  private TextField valueField = new TextField("" + STRONG, 4);
  private Choice valueChoice = new Choice();

  StrengthHandler()
  {
    valueChoice.add("REQUIRED");
    valueChoice.add("STRONG");
    valueChoice.add("MEDIUM");
    valueChoice.add("WEAK");
    valueChoice.select(1);
    int width = valueChoice.getSize().width;

    setLayout(new GridLayout());
    add("West", valueField);
    add("Center", valueChoice);

    valueChoice.addItemListener(this);
  }

  public void itemStateChanged(ItemEvent e)
  {
    // ** handle clicking button by cycling through presets
    String str = (String) e.getItem();

    int strength = getStrength(); // default strength to old strength

    if (str.equals("REQUIRED")) {
      strength = REQUIRED;
    } else if (str.equals("STRONG")) {
      strength = STRONG;
    } else if (str.equals("MEDIUM")) {
      strength = MEDIUM;
    } else if (str.equals("WEAK")) {
      strength = WEAK;
    }

    valueField.setText("" + strength);
  }

  // ** Returns a value from the strength text field
  // ** (If a number is stored then that value is returned
  // **  If a recognised word is stored then it is converted to a value
  // **  Otherwise, the value for STRONG is returned)
  public int getStrength()
  {
    int strength;

    String str = new String(valueField.getText());

    // get Intger value corresponding to text
    Integer val = Parser.getValue(str);
    if (val != null) {
      // ** a legitamate value is stored in the text field
      strength = val.intValue();
    } else {
      // ** something other than an integer value was entered
      strength = STRONG; // change to strong
      valueField.setText("" + strength); // change text field too
    }
    return strength;
  }
}

class ConHandler extends List implements ActionListener
{
  private DeltaBlueClient deltaClnt;
  private Vector userCons = new Vector(); // vector of Input CONstraintS

  ConHandler(DeltaBlueClient deltaClnt)
  {
    super(7, true);
    this.deltaClnt = deltaClnt;

    addActionListener(this);
  }

  // ** handle double clicking on a constraint in the constraints list
  public void actionPerformed(ActionEvent e)
  {
    // ** get index of clicked constraint
    int index = getSelectedIndex(); // use currently selected index by default
    if (index < 0) {
      // ** the item must have been deselected already
      String str = e.getActionCommand(); // get string item
      String [] items = getItems();
      for (int i = 0; i < items.length; i++) {
        if (str.equals(items[i])) {
          index = i;
          break;
        }
      }
    }

    if (index >= 0) {
      // remove selected constraint from list
      deltaClnt.removeUserConstraint(index);
    }
  }

  public void reset()
  {
    removeAll(); // remove all items from list
    userCons.removeAllElements(); // no input constraints yet!
  }

  // add constraint named "str" to the list of variables
  public void addCon(UserConstraint userCon)
  {
    // get strength of input constraint
    int strength = userCon.getStrength();

    // search through the existing list so that we keep the order of strength
    int i = 0;
    while (i < userCons.size() &&
           strength < ((UserConstraint) userCons.elementAt(i)).getStrength()) {
      i++;
    }
    userCons.insertElementAt(userCon, i);

    update(); // update the visible list of constraints
  }

  // remove constraint with corres. "index" from the list of variables
  public UserConstraint removeCon(int index)
  {
    // ** remove selected constraint
    
    UserConstraint removed = (UserConstraint) userCons.elementAt(index);

    userCons.removeElementAt(index); // remove element from main vector

    update(); // update the visible list of constraints

    return removed;
  }

  // ** change the first selected edit constraint's value
  public boolean changeValue(int value)
  {
    for (int i = 0; i < getItemCount(); i++) {
      if (isIndexSelected(i)) {
	UserConstraint userCon = (UserConstraint) userCons.elementAt(i);
	if (userCon.isEdit()) {
	  // ** we have a selected edit constraint
	  ((EditCon) ((Vector) (userCon.getConstraints())).elementAt(0))
	    .changeValue(value);
	  return true;
	}
      }
    }

    return false;
  }
  
  public void update()
  {
    removeAll(); // remove all variables from visible list

    for (Enumeration e = userCons.elements() ; e.hasMoreElements() ;) {
      UserConstraint userCon = (UserConstraint) (e.nextElement());

      // ** are all the sub-constraints satisfied or not?
      char sat = 't'; // default to all subconstraints satisfied
      Vector cons = userCon.getConstraints();
      for (int j = 0; j < cons.size(); j++) {
        Constraint con = (Constraint) cons.elementAt(j);
        if (((Constraint) cons.elementAt(j)).selectedMethod == null) {
          // ** at least one not satisfied
          sat = 'f';
          break;
        }
      }

      addItem("{" + userCon.getStrength() + ", " +sat+ "} "+userCon.getName());
    }
  }

  // ** get an array containing all the selected constraints (if there are any)
  // ** or all of the constraints (if none are selected)
  public Constraint [] getSelectedCons()
  {
    int [] useIndexes = getSelectedIndexes();
    if (useIndexes.length == 0) {
      // ** no constraints selected so we'll use all of them
      useIndexes = new int[getItemCount()];
      for (int i = 0; i < useIndexes.length; i++) {
	useIndexes[i] = i;
      }
    }
    // ** get all the constraints' sub-constraints
    Vector selConsVec = new Vector();
    for (int i = 0; i < useIndexes.length; i++) {
      Vector cs = ((UserConstraint) userCons.elementAt(useIndexes[i]))
        .getConstraints();
      for (int j = 0; j < cs.size(); j++) {
        selConsVec.addElement((Constraint) cs.elementAt(j));
      }
    }

    Constraint [] selCons = new Constraint[selConsVec.size()];
    selConsVec.copyInto(selCons);
    return selCons;
  }
}

class VarHandler extends List
{
  private Hashtable varHash = new Hashtable();

  VarHandler()
  { super(7, true); }

  public void reset()
  {
    removeAll(); // remove all variables from list
    varHash.clear(); // remove all variables from hashtable
  }

  public Variable addVar(String name)
  {
    Variable var;

    // check whether we need to create a new variable
    if (varHash.containsKey(name)) {
      // ** already exists
      var = (Variable) varHash.get(name); // get existing variable
    } else {
      // ** does not exist
      var = new Variable(0); // make new variable
      varHash.put(name, var); // add to variable hashtable
    }

    return var;
  }

  public void update()
  {
    removeAll(); // remove all variables from visible list

    for (Enumeration e = varHash.keys() ; e.hasMoreElements() ;) {
      String name = (String) (e.nextElement());
      Variable v = (Variable) (varHash.get(name));
      if (v.cons.isEmpty()) {
        // variable is not tied to any constraints, so remove it
        varHash.remove(name);
      } else {
        // ** variable is part of the graph so add it to the visible list
        char stay = 'f'; // default "stay" char to false
        if (v.stay) { stay = 't';}
        addItem("{" + v.walkStrength +", "+stay+"} " + name + " = " + v.value);
      }
    }
  }

  public Variable [] getSelectedVars()
  {
    String [] selItems = getSelectedItems();
    Variable [] selVars = new Variable[selItems.length];
    for (int i = 0; i < selItems.length; i++) {
      String str = selItems[i]; // get string at position in list
      // get variable name from list entry (between "}" and "=")
      String varName = str.substring(str.indexOf("}") + 1, str.indexOf("="))
        .trim();
      selVars[i] = (Variable) varHash.get(varName);
    }

    return selVars;
  }
}

class UserConstraint
{
  private String name;
  private Vector cons;
  private int strength;

  UserConstraint(String name, Vector cons, int strength)
  {
    this.name = name;
    this.cons = cons;
    this.strength = strength;
  }

  public String getName()
  { return name; }
  
  public Vector getConstraints()
  { return cons; }

  public int getStrength()
  { return strength; }

  // ** return whether this is an edit constraint
  public boolean isEdit()
  {
    // get first constraint
    if ((Constraint) cons.elementAt(0) instanceof EditCon) {
      return true;
    } else {
      return false;
    }
  }
}

class TableOfConstraints extends Hashtable
{
  TableOfConstraints()
  {
    super();

    // ** Start off with the table containing all the deltablue constraints
    // ** corresponding to the built in operators used by the parser

    // ** constraints corresponding to relational operators
    put(TableOfOperators.EQU, "EqualityCon");
    // ** constraints corresponding to standard unary operators
    put(TableOfOperators.FAC, "FactorialCon");
    put(TableOfOperators.NEG, "NegationCon");
    // ** constraints corresponding to standard binary operators
    put(TableOfOperators.ADD, "AdditionCon");
    put(TableOfOperators.SUB, "SubtractionCon");
    put(TableOfOperators.MUL, "MultiplicationCon");
    put(TableOfOperators.DIV, "DivisionCon");
  }

  public boolean put(int oprType, String className)
  {
    try {
      super.put(new Integer(oprType), Class.forName(className));
    }
    catch (ClassNotFoundException e) {
      System.out.println
        ("Unable to find constraint class \"" + className + "\"!");
      return false; // addition failed
    }
    return true; // addition successful
  }
}

// ** Handles the generation of sub-constraints from one main constraint
// ** given in the form an expression.  Also, tells the DeltaBlue Client of
// ** any new variables it needs to know about
class ConGenerator implements Visitor
{
  private VarHandler varHdlr;
  private TableOfConstraints conTable;
  private Vector cs; // vector to hold all subconstraints
  private int strength; // strength for all subconstraints

  ConGenerator(VarHandler varHdlr, TableOfConstraints conTable)
  {
    this.varHdlr = varHdlr; // get the handler for input variables
    this.conTable = conTable;
  }

  // ** given an expression, return the sub-constraints which make it up
  // ** (each sub-constraint is a basic constraint like "addition", "equality")
  public Vector getSubConstraints(Expression e, int strength)
  {
    this.strength = strength;

    cs = new Vector(); // no subconstraints yet

    e.visit(this); // traverse the entire expression tree

    return cs; // return vector of sub-constraints
  }

  public Object visitVar(VarExpr e)
  {
    // ** visiting a variable expression

    // tell the handler that we have stumbled across a "user" variable
    // The handler will return a variable object that corresponds to this
    return varHdlr.addVar(e.getName());
  }

  public Object visitNum(NumExpr e)
  {
    // ** visiting a constant number expression -this will be a stay constraint
    
    Variable [] vars = new Variable [1];
    vars[0] = new Variable(0);
    
    CompleteConstraint con = new SetCon(e.getValue());
    con.init(vars, strength, false);
    cs.addElement(con);

    return vars[0];
  }

  // ** visiting an operation expression (also handles relational ones)
  public Object visitOpr(OprExpr e)
  {
    Operator op = e.getOp(); // get operator
    int opType = op.getType(); // type of operator
    int opArity = op.getArity(); // arity of operator
    Expression [] exprs = e.getExprs(); // get all argument expressions

    Variable output; // output variable to be returned

    // ** define array for variables associated with the operation expression
    Variable [] vars;

    if (op.isRelational()) {
      // one variable for each argument
      vars = new Variable[opArity];
      output = null; // no output variable
    } else {
      // ** a standard operation expression with an output
      // one variable for each argument plus one for the output
      vars = new Variable[opArity + 1];
      output = new Variable(0); // create output variable
      // final variable is the default output
      vars[vars.length - 1] = output;
    }

    // ** get variables representing the outputs from each argument expression
    for (int i = 0; i < exprs.length; i++) {
      vars[i] = (Variable) exprs[i].visit(this);
    }

    if (conTable.containsKey(new Integer(opType))) {
      // ** this operator does indeed have a deltablue constraint class for it
      try {
        CompleteConstraint con = (CompleteConstraint)
          ((Class) conTable.get(new Integer(opType))).newInstance();
        con.init(vars, strength, false);
        cs.addElement(con);
      }
      catch (Exception except) {
        System.out.println("Error Instantiating Class");
      }
    }

    return output;
  }
}

// ** More Complete Constraints

// ** vars[0] = number which can be changed using "changeValue()" method
class EditCon extends CompleteConstraint
{
  protected int value = 0; // set value to 0 by default

  // ** force this constraint to be an input constraint
  public void init(Variable [] vars, int strength, boolean inputFlag)
  { super.init(vars, strength, true); }

  public int methodCode(int number)
  { return value; }

  public void changeValue(int newValue)
  { value = newValue; }
}

// **** Example Constraints ****

// ** centigrade - fahrenheit converter
class TempConvCon extends CompleteConstraint
{
  public int methodCode(int number)
  {
    if (number == 0) {
      return (5 * (vars[1].value - 32)) / 9;
    } else {
      return (9 * vars[0].value) / 5 + 32;
    }
  }
}


/*

The End

*/
