/*

Dissertation - "Using Java To Implement Constraint Satisfaction Systems"
========================================================================

Implementation Of The Core DeltaBlue Algorithm And Data Structures
------------------------------------------------------------------

Fully Optimised Version.

Compatible With Java Version 1.1.3

This code is largely based on the pseudocode given in the appendix to
"Multi-way versus One-way Constraints in User Interfaces: Experience with the
DeltaBlue Algorithm" article in "Software-Practice and Experience" Vol 23(5).

Code which does differ significantly from the pseudocode is cleary commented
as such.

*/

import java.util.*;

class DeltaBlue
{
  // ** strengths
  public static int WEAKEST = 0;
  public static int REQUIRED = 1000;

  // this stringbuffer is used to return error messages from the algorithm to
  // the client - not explicitly specified in the pseudocode
  private StringBuffer dbMessage = new StringBuffer();

  private MarkHandler markHdlr = new MarkHandler(); // gives next mark value

  // *** Entry Points ***

  public String addConstraint(Constraint c)
  {
    //System.out.println("addConstraint()");

    // **** Start of deviation from the pseudocode ****
    // ** make sure that the strength of the new constraint is greater
    // ** than WEAKEST and less than or equal to REQUIRED by changing the
    // ** existing value to the nearest "legal" value if outside the range.
    if (c.strength <= WEAKEST) {
      c.strength = WEAKEST + 1;
    } else if (c.strength > REQUIRED) {
      c.strength = REQUIRED;
    }
    // **** End of deviation from the pseudocode ****

    dbMessage.setLength(0); // clear message
    c.selectedMethod = null; // no method selected yet
    
    // ** tell corresponding variables about new constraint
    Variable [] vars = c.vars; // get array of variables
    for (int i = 0; i < vars.length; i++) {
      vars[i].cons.addElement(c);
    }

    incrementalAdd(c);

    return dbMessage.toString();
  }

  public void removeConstraint(Constraint c)
  {
    //System.out.println("removeConstraint()");

    if (enforced(c)) {
      incrementalRemove(c);
    } else {
      Variable [] vars = c.vars;
      for (int i = 0; i < vars.length; i++) {
        vars[i].cons.removeElement(c);
      }
    }
  }

  // ** extraxt plan from variables
  public Plan extractPlan(Variable [] vars)
  {
    //System.out.println("extractPlan() (from variables)");

    Queue sources = new Queue();
    int varsLength = vars.length;
    for (int i = 0; i < varsLength; i++) {
      Variable v = vars[i];
      ConVector cons = v.cons;
      for (int j = 0; j < cons.size(); j++) {
        Constraint c = (Constraint) cons.elementAt(j);
        if (c.inputFlag && enforced(c)) {
          sources.add(c);
        }
      }
    }

    return makePlan(sources);
  }

  // ** extraxt plan from constraints
  public Plan extractPlan(Constraint [] cons)
  {
    //System.out.println("extractPlan() (from constraints)");

    Queue sources = new Queue();
    int consLength = cons.length;
    for (int i = 0; i < consLength; i++) {
      Constraint c = cons[i];
      if (c.inputFlag && enforced(c)) {
        sources.add(c);
      }
    }

    return makePlan(sources);
  }

  public void executePlan(Plan pl)
  {
    //System.out.println("executePlan()");

    // ** execute selected method for each constraint in plan (in order)
    int planSize = pl.size();
    for (int i = 0; i < planSize; i++) {
      Method m = ((Constraint) pl.elementAt(i)).selectedMethod;
      m.output.value = m.code(); // execute code and assign it to output
    }
  }

  // **** Adding Constraints ****

  private void incrementalAdd(Constraint c)
  {
    //System.out.println("incrementalAdd(Constraint)");

    int mark = markHdlr.newMark();
    Constraint retracted = enforce(c, mark);
    while (retracted != null) {
      retracted = enforce(retracted, mark);
    }
  }
  
  private Constraint enforce(Constraint c, int mark)
  {
    //System.out.println("enforce(Constraint)");
    selectMethod(c, mark);
    if (enforced(c)) {
      // ** given constraint is enforced

      // ** mark all the inputs to the given constraint
      Variable [] vars = c.vars;
      for (int i = 0; i < vars.length; i++) {
        Variable v = vars[i];
        if (v != c.selectedMethod.output) {
          // ** this ones not the current output so mark it
          v.mark = mark;
        }
      }

      Constraint retracted = c.selectedMethod.output.determinedBy;
      if (retracted != null) {
        retracted.selectedMethod = null;
      }
      c.selectedMethod.output.determinedBy = c;
      if (!addPropagate(c, mark)) {
        dbMessage.append("Cycle Encountered!");
        return null;
      }
      c.selectedMethod.output.mark = mark;
      return retracted;
    } else {
      if (c.strength == REQUIRED) {
        dbMessage.append("Failed to enforce a required constraint!");
      }
      return null;
    }
  }

  private void selectMethod(Constraint c, int mark)
  {
    //System.out.println("selectMethod()");
    c.selectedMethod = null;
    int bestOutStrength = c.strength;
    
    Method [] methods = c.methods;

    for (int i = 0; i < methods.length; i++) {
      Method m = methods[i];
      if (m.output.mark != mark &&
          weaker(m.output.walkStrength, bestOutStrength)) {
        c.selectedMethod = m;
        bestOutStrength = m.output.walkStrength;
        //System.out.println("Selected Method " + i);
      }
    }
  }

  // ** Recompute the walkabout strengths, the stay flags, and possibly the
  // ** values of all variables downstream of the given constraint
  private boolean addPropagate(Constraint c, int mark)
  {
    //System.out.println("addPropagate()");
    Queue todo = new Queue();
    todo.add(c);

    while (!todo.isEmpty()) {
      Constraint d = (Constraint) todo.remove(); // get oldest element in todo
      if (d.selectedMethod.output.mark == mark) {
        incrementalRemove(c);
        return false;
      }
      recalculate(d);

      // ** add d's output variable's consuming constraints to "todo"
      Variable out = d.selectedMethod.output;
      ConVector cons = out.cons;
      for (int i = 0; i < cons.size(); i++) {
        Constraint con = (Constraint) cons.elementAt(i);
        if (enforced(con) && con != out.determinedBy) {
          todo.add(con);
        }
      }
    }
    return true;
  }

  // **** Removing Constraints ****

  private void incrementalRemove(Constraint c)
  {
    Variable out = c.selectedMethod.output;
    c.selectedMethod = null;

    Variable [] vars = c.vars;
    for (int i = 0; i < vars.length; i++) {
      vars[i].cons.removeElement(c);
    }

    Constraint strongestEligible = removePropagateFrom(out);
    if (strongestEligible != null) {
      // ** a "real" constraint is eligible to be added
      incrementalAdd(strongestEligible);
    }
  }

  private Constraint removePropagateFrom(Variable out)
  {
    // strongest eligible constraint
    Constraint strongestEligible = null;
    // strength of strongest eligible constraint
    // (default to a value lower than the weakest)
    int strongestStrength = WEAKEST - 1;

    out.determinedBy = null;
    out.walkStrength = WEAKEST;
    out.stay = true;

    Queue todo = new Queue();
    todo.add(out);

    while (!todo.isEmpty()) {
      Variable v = (Variable) todo.remove(); // get oldest element in todo

      ConVector cons = v.cons;
      for (int i = 0; i < cons.size(); i++) {
        Constraint c = (Constraint) cons.elementAt(i);
        if (!enforced(c)) {
          // ** constraint not enforced so add it
          // ** to unenforced downstream list
          if (c.strength > strongestStrength && eligible(c)) {
            // ** this is the strongest unenforced eligible
            // ** constraint so far
            strongestEligible = c;
            strongestStrength = c.strength;
          }
        } else if (c != v.determinedBy) {
          // ** constraint is enforced and does not determine output variable
          recalculate(c);
          todo.add(c.selectedMethod.output);
        }
      }
    }

    return strongestEligible;
  }

  // **** Extracting Plans ****

  private Plan makePlan(Queue sources)
  {
    //System.out.println("makePlan(" + sources.size() + " Constraints)");
    Plan pl = new Plan();
    int mark = markHdlr.newMark();
    Queue hot = sources;

    while (!hot.isEmpty()) {
      Constraint c = (Constraint) hot.remove(); // get oldest element in todo

      if (c.selectedMethod.output.mark != mark && inputsKnown(c, mark)) {
        pl.addElement(c); // add constraint c to plan
        c.selectedMethod.output.mark = mark;
        // ** add all consuming constraints to hot
        Variable out = c.selectedMethod.output;
        ConVector cons = out.cons;
        for (int i = 0; i < cons.size(); i++) {
          Constraint con = (Constraint) cons.elementAt(i);
          if (enforced(con) && con != out.determinedBy) {
            hot.add(con);
          }
        }
      }
    }

    return pl;
  }

  // **** General Purpose Methods ****

  // ** returns true iff the given constraint is stronger than
  // ** at least one of its potential outputs
  private boolean eligible(Constraint c)
  {
    Method [] methods = c.methods;
    for (int i = 0; i < methods.length; i++) {
      if (c.strength > methods[i].output.walkStrength) {
        return true;
      }
    }
    return false;
  }

  private boolean inputsKnown(Constraint c, int mark)
  {
    Variable [] vars = c.vars;
    for (int i = 0; i < vars.length; i++) {
      Variable v = vars[i];
      if (v != c.selectedMethod.output) {
        // ** this one is not the given constraint's output
        if (!(v.mark == mark || v.stay)) {
          return false;
        }
      }
    }

    return true;
  }

  // ** updates the walkabout strength and the stay flag of the given
  // ** constraints output variable.  If the stay flag is true then the value
  // ** of the output variable is computed
  private void recalculate(Constraint c)
  {
    //System.out.println("recalculate()");

    Variable out = c.selectedMethod.output; // get output variable
    out.walkStrength = outputWalkStrength(c);
    out.stay = constantOutput(c);
    if (out.stay) {
      out.value = c.selectedMethod.code();
      //System.out.println("Recalculated a value " + out.value);
    }
  }

  private int outputWalkStrength(Constraint c)
  {
    int minStrength = c.strength;
    Method [] methods = c.methods;
    for (int i = 0; i < methods.length; i++) {
      Method m = methods[i];
      if (m.output != c.selectedMethod.output &&
          weaker(m.output.walkStrength, minStrength)) {
        minStrength = m.output.walkStrength;
      }
    }
    return minStrength;
  }

  private boolean constantOutput(Constraint c)
  {
    if (c.inputFlag) {
      return false;
    }
    Variable [] vars = c.vars;
    for (int i = 0; i < vars.length; i++) {
      Variable v = vars[i];
      if (v != c.selectedMethod.output && !v.stay) {
        // ** not the output of the given constraint and stay flag false
        return false;
      }
    }
    return true;
  }

  private boolean enforced(Constraint c)
  { return (c.selectedMethod != null); }

  private boolean weaker(int s1, int s2)
  { return (s1 < s2); }

  class MarkHandler
  {
    private int nextMark = 1; // next free mark value to be used
    
    public int newMark()
    {
      //System.out.println("Mark " + nextMark);
      return nextMark++;
    }
  }
}

// **** Data Structures ****

class ConVector extends Vector
{
  ConVector()
  { super(20, 5000); }
}

// ** a plan is simply a vector of constraints
class Plan extends ConVector
{
  Plan()
  { super(); }
}

class Variable
{
  public int value;
  public ConVector cons = new ConVector();
  public Constraint determinedBy;
  public int walkStrength = DeltaBlue.WEAKEST;
  public int mark = 0;
  public boolean stay = true;

  Variable(int value)
  { this.value = value; }
}

abstract class Constraint
{
  public Variable [] vars;
  public int strength;
  public boolean inputFlag;
  public Method [] methods;
  public Method selectedMethod;
}

abstract class Method
{
  public Variable output;

  abstract public int code();
}


/*

The End

*/
