/*

Dissertation - "Using Java To Implement Constraint Satisfaction Systems"
========================================================================

Basic DeltaBlue Complete Constraints
------------------------------------

Defines an abstract class for "Complete" constraints and the corresponding
"complete" method class.  All the common constraints are also defined.

Compatible With Java Version 1.1.3

*/

// ** "Complete" constraints which define the method code inside the class
abstract class CompleteConstraint extends Constraint
{
  public void init(Variable [] vars, int strength, boolean inputFlag)
  {
    this.vars = vars;
    this.strength = strength;
    this.inputFlag = inputFlag;

    methods = new CompleteMethod[vars.length];

    // one method for each possible output variable (by default)
    for (int i = 0; i < vars.length; i++) {
      methods[i] = new CompleteMethod(this, i, vars[i]);
    }
  }

  abstract public int methodCode(int number);
}

// ** Methods for "Complete" Constraints
class CompleteMethod extends Method
{
  private CompleteConstraint c;
  private int number;

  CompleteMethod(CompleteConstraint c, int number, Variable output)
  {
    this.c = c;
    this.number = number;
    this.output = output;
  }

  public int code()
  { return c.methodCode(number); }
}

// **** Standard Non-Operator Complete Constraints ****
// (these ones need an extra argument on top of those allowed by "init" method)

// ** vars[0] = number
class SetCon extends CompleteConstraint
{
  protected int value;

  SetCon(int value)
  { this.value = value; }

  public int methodCode(int number)
  { return value; }
}

// **** Standard Operator Complete Constraints ****

// constrain that a variable's value should not be changed
class StayCon extends CompleteConstraint
{
  public int methodCode(int number)
  { return vars[0].value; }
}

// ** vars[0] ! = vars[1]
class FactorialCon extends CompleteConstraint
{
  public int methodCode(int number)
  {
    if (number == 0) {
      // ** backwards direction
      int value = -1; //default value of vars[0] if inverse factorial not exist
      int temp = vars[1].value;
      if (temp >= 1) {
        int i;
        for (i = 1; temp % i == 0; i++) {
          temp /= i;
        }
        if (temp == 1) {
          // the inverse factorial does indeed exist!
          value = i - 1;
        }
      }
      return value;
    } else {
      // ** forwards direction
      // ** return true value if vars[0] >= 0, otherwise return 0
      if (vars[0].value >= 0) {
        int value = 1;
        for (int i = vars[0].value; i > 1; i--) {
          value *= i;
        }
        return value;
      } else {
        return 0;
      }
    }
  }
}

// ** - vars[0] = vars[1];
class NegationCon extends CompleteConstraint
{
  public int methodCode(int number)
  {
    if (number == 0) {
      return -vars[1].value;
    } else {
      return -vars[0].value;
    }
  }
}

// ** vars[0] = vars[1]
class EqualityCon extends CompleteConstraint
{
  public int methodCode(int number)
  {
    if (number == 0) {
      return vars[1].value;
    } else {
      return vars[0].value;
    }
  }
}

// ** vars[0] + vars[1] = vars[2]
class AdditionCon extends CompleteConstraint
{
  public int methodCode(int number)
  {
    if (number == 0) {
      return vars[2].value - vars[1].value;
    } else if (number == 1) {
      return vars[2].value - vars[0].value;
    } else {
      return vars[0].value + vars[1].value;
    }
  }
}

// ** vars[0] - vars[1] = vars[2]
class SubtractionCon extends AdditionCon
{
  public void init(Variable [] vars, int strength, boolean inputFlag)
  {
    // "vars[0] - vars[1] = vars[2]" is like "vars[2] + vars[1] = vars[0]"
    // ** swap variables 0 and 2...
    Variable dummy = vars[0];
    vars[0] = vars[2];
    vars[2] = dummy;
    // ... and use the addition constraint
    super.init(vars, strength, inputFlag);
  }
}

// ** vars[0] * vars[1] = vars[2]
class MultiplicationCon extends CompleteConstraint
{
  public int methodCode(int number)
  {
    if (number == 0) {
      return vars[2].value / vars[1].value;
    } else if (number == 1) {
      return vars[2].value / vars[0].value;
    } else {
      return vars[0].value * vars[1].value;
    }
  }
}

// ** vars[0] / vars[1] = vars[2]
class DivisionCon extends MultiplicationCon
{
  public void init(Variable [] vars, int strength, boolean inputFlag)
  {
    // "vars[0] / vars[1] = vars[2]" is like "vars[2] * vars[1] = vars[0]"
    // ** swap variables 0 and 2...
    Variable dummy = vars[0];
    vars[0] = vars[2];
    vars[2] = dummy;
    // ... and use the multiplication constraint
    super.init(vars, strength, inputFlag);
  }
}


/*

The End

*/
